<?php

namespace App\Http\Controllers;

use App\Models\Member;
use App\Models\Transaction;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;


class DashboardController extends Controller
{
    /**
     * Display the dashboard with various statistics and data.
     */
    public function index()
    {
        $now = Carbon::now();
        $month = $now->format('m');
        $year = $now->format('Y');

        $dashboard_time = $now->format('l, h:i A');
        $dashboard_date = $now->format('F d, Y');
        $currency_symbol = currency_symbol();
        $earnings = $this->earnings($month, $year);
        $count_members = $this->count_members($month, $year);
        $total_members = $count_members['total'];
        $new_members = $count_members['new'];

        $week_dates = week_dates($now->format('Y-m-d')); 
        $date1 = $week_dates['date1']; 
        $date2 = $week_dates['date2'];

        $chart_data = $this->generate_chart($date1, $date2);

        $latest_members = $this->latest_members();
        $latest_transactions = $this->latest_transactions();

        // KEEP IT IN THE END FOR THE DATE FORMAT PURPOSE.
        $date1 =  $date1->format('d-m-Y');
        $date2 = $date2->format('d-m-Y');

        return view('dashboard', compact(
            'dashboard_time',
            'dashboard_date',
            'earnings', 
            'total_members', 
            'new_members',
            'chart_data',
            'currency_symbol',
            'date1',
            'date2',
            'latest_members',
            'latest_transactions',
        ));
    }

    /**
     * Generate chart data based on transactions within a given date range.
     *
     * @param  string  $date1  The start date of the date range.
     * @param  string  $date2  The end date of the date range.
     * @return array           The chart data containing day-wise transaction totals.
     */
    private function generate_chart($date1, $date2) {
        $chart_query = DB::table('transactions')
        ->select(DB::raw('DAYNAME(date) as day, SUM(amount) as total, dr_cr'))
        ->whereBetween('date', [$date1, $date2])
        ->groupBy('day', 'dr_cr')
        ->orderBy('day')
        ->get();

        $chart_data = $this->generate_chart_data($chart_query);

        return $chart_data;
    }

    /**
     * Generate chart data based on transactions.
     *
     * @param  \Illuminate\Support\Collection  $transactions  The transactions to be processed.
     * @return array The processed chart data containing income and expense totals.
     */
    private function generate_chart_data($transactions): array
    {
        $data = [
            'income'=> [],
            'expense'=> [],
        ];

        $income = [
            'Sunday' => 0,
            'Monday' => 0,
            'Tuesday' => 0,
            'Wednesday' => 0,
            'Thursday' => 0,
            'Friday' => 0,
            'Saturday' => 0,
        ];
        $expense = [
            'Sunday' => 0,
            'Monday' => 0,
            'Tuesday' => 0,
            'Wednesday' => 0,
            'Thursday' => 0,
            'Friday' => 0,
            'Saturday' => 0,
        ];
 
        // POPULATE THE ARRAYS WITH THE PAYMENT TOTALS
        foreach ($transactions as $transaction) {
            $day = $transaction->day;
            $total = $transaction->total;
            $type = $transaction->dr_cr;

            if($type == 'cr') {
                $income[$day] +=  $total;     
            } elseif($type == 'dr') {
                $expense[$day] +=  $total; 
            }
        }

        // CONVERT THE ARRAYS TO SIMPLE ARRAYS WITH NUMERIC KEYS
        $income = array_values($income);
        $expense = array_values($expense);

        $data['income'] = $income;
        $data['expense'] = $expense;

        return $data;
    }

    /**
     * Count the total number of members and new members for a given month and year.
     *
     * @param  int  $month  The month for which to count new members.
     * @param  int  $year   The year for which to count new members.
     * @return array        An array containing the total number of members and new members.
     */
    private function count_members($month, $year) {
        $total = Member::count();

        $new_members = Member::whereMonth('created_at', $month)
            ->whereYear('created_at', $year)
            ->count();

        return ['total' => $total, 'new' => $new_members];
    }

    /**
     * Calculate the earnings for a given month and year.
     *
     * @param  int  $month  The month for which to calculate earnings.
     * @param  int  $year   The year for which to calculate earnings.
     * @return float        The total earnings for the specified month and year.
     */
    private function earnings($month, $year) { 
        $earnings = Transaction::whereMonth('date', $month)
            ->whereYear('date', $year)
            ->where('dr_cr', 'cr')
            ->sum('amount');
        
        return $earnings;
    }

    /**
     * Retrieve the latest members, ordered by their creation date.
     *
     * @return \Illuminate\Support\Collection  The collection of latest members.
     */
    private function latest_members() { 
        $members = Member::orderBy('created_at', 'desc')
            ->take(20)
            ->get();

        return $members;
    }

    /**
     * Retrieve the latest transactions, ordered by date.
     *
     * @return \Illuminate\Support\Collection  The collection of latest transactions.
     */
    private function latest_transactions() { 
        $transactions = Transaction::orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        return $transactions;
    }
}